//
//  SPBoxView.m
//  SamplePlugIn
//
//  Created by Kai on 11.1.08.
//  Copyright 2008 __MyCompanyName__. All rights reserved.
//

#include <cmath>
#include <algorithm>

#import "SPBoxView.h"
#import "SPLinearGradient.h"


//-------------------------------------------------------------------------------------------
@implementation SPBoxView
//-------------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------------
- (id) initWithFrame:(NSRect)frame {
	self = [super initWithFrame:frame];
	if (self) {
		_cornerRadius = 3.0;
		_borderWidth = 0.25;
		
		_fillColor = [[NSColor whiteColor] retain];
		_borderColor = [[NSColor blackColor] retain];
	}
	return self;
}

//-------------------------------------------------------------------------------------------
- (void) dealloc {
	[_fillColor release];
	[_borderColor release];
	[_gradient release];
	[_boxShadow release];
	
	[super dealloc];
}

//-------------------------------------------------------------------------------------------
- (CGFloat) cornerRadius { return _cornerRadius; }

//-------------------------------------------------------------------------------------------
- (void) setCornerRadius:(CGFloat)value {
	if (value != _cornerRadius) {
		_cornerRadius = value;
		[self setNeedsDisplay:YES];
	}
}

//-------------------------------------------------------------------------------------------
- (NSColor*) fillColor { return [[_fillColor retain] autorelease]; }

//-------------------------------------------------------------------------------------------
- (void) setFillColor:(NSColor*)value {
	if (value != _fillColor) {
		[_fillColor release];
		_fillColor = [value copy];		// TODO: validate that copy just retains due to immutability
		[self setNeedsDisplay:YES];
	}
}

//-------------------------------------------------------------------------------------------
- (SPLinearGradient*) gradient { return [[_gradient retain] autorelease]; }

//-------------------------------------------------------------------------------------------
- (void) setGradient:(SPLinearGradient*)value {
	if (value != _gradient) {
		[_gradient release];
		_gradient = [value retain];	// may want to copy instead
		[self setNeedsDisplay:YES];
	}
}

//-------------------------------------------------------------------------------------------
- (NSColor*) borderColor { return [[_borderColor retain] autorelease]; }

//-------------------------------------------------------------------------------------------
- (void) setBorderColor:(NSColor*)value; {
	if (value != _borderColor) {
		[_borderColor release];
		_borderColor = [value copy];		// TODO: validate that copy just retains due to immutability
		[self setNeedsDisplay:YES];
	}
}

//-------------------------------------------------------------------------------------------
- (CGFloat) borderWidth { return _borderWidth; }

//-------------------------------------------------------------------------------------------
- (void) setBorderWidth:(CGFloat)value {
	if (value != _borderWidth) {
		_borderWidth = value;
		[self setNeedsDisplay:YES];
	}
}

//-------------------------------------------------------------------------------------------
- (NSShadow*) boxShadow { return [[_boxShadow retain] autorelease]; }

//-------------------------------------------------------------------------------------------
- (void) setBoxShadow:(NSShadow*)value {
	if (value != _boxShadow) {
		[_boxShadow release];
		// Note: NSShadow is clearly not immutable, therefore copy will always create a real
		// copy. That’s unfortunate, but I want to make sure no surprising things happen due to
		// just retaining the input value.
		_boxShadow = [value copy];
		[self setNeedsDisplay:YES];
	}
}

//-------------------------------------------------------------------------------------------
- (void) invalidateInsets { _insetsAreValid = NO; }

//-------------------------------------------------------------------------------------------
- (void) calculateInsets {
	// Important: SPBoxView (IBIntegration) depends on being able to overwrite boxShadow to
	// do its outlet trick.
	NSShadow* shadow = [self boxShadow];
	
	// Calculate insets.
	NSSize shadowOffset = NSZeroSize;
	CGFloat blurRadius = 0.0;
	if (shadow) {
		shadowOffset = [shadow shadowOffset];
		blurRadius = [shadow shadowBlurRadius];
		//TODO: convert to default user coordinates.
	}
	// Shows that it is better to have one more pixel leeway for unclipped drawing of the
	// border.
	CGFloat borderMargin = (_borderColor ? _borderWidth / 2.0 : 0.0) + 1.0;
	
	_leftInset   = std::ceil (std::max (blurRadius - shadowOffset.width,  borderMargin));
	_rightInset  = std::ceil (std::max (blurRadius + shadowOffset.width,  borderMargin));
	_bottomInset = std::ceil (std::max (blurRadius - shadowOffset.height, borderMargin));
	_topInset    = std::ceil (std::max (blurRadius + shadowOffset.height, borderMargin));
}

//-------------------------------------------------------------------------------------------
- (void) drawRect:(NSRect)rect {
	// Important: SPBoxView (IBIntegration) depends on being able to overwrite boxShadow and
	// gradient to do its outlet trick.
	// Additionally, it does currently needs the shadow lookup to happen before
	// _insetsAreValid is tested.
	NSShadow* shadow = [self boxShadow];
	SPLinearGradient* gradient = [self gradient];
	
	// Make sure we have valid insets.
	if (! _insetsAreValid) {
		[self calculateInsets];
		_insetsAreValid = YES;
	}
	
	NSRect box = [self bounds];
	
	box.origin.x += _leftInset;
	box.size.width -=  _leftInset + _rightInset;

	box.origin.y += [self isFlipped] ? _topInset : _bottomInset;
	box.size.height -= _topInset + _bottomInset;
	
	//TODO: should use Quartz traqnsparency layer to apply the shadow to fill plus stroke.
	// For now I apply it to the fill only.
	if (shadow) {
		[NSGraphicsContext saveGraphicsState];
		[shadow set];
	}
	
	NSBezierPath* path;
	if (_cornerRadius == 0.0)
		path = [NSBezierPath bezierPathWithRect:box];
	else
		path = [NSBezierPath bezierPathWithRoundedRect:box xRadius:_cornerRadius yRadius:_cornerRadius];

	if (gradient) {
		// Following seems to be necessary.
		//TODO: think about alpha in gradient.
		if (shadow) {
			[[NSColor whiteColor] setFill];
			[path fill];
		}
		[gradient drawInBezierPath:path/* angle:_gradiantAngle*/];
	} else if (_fillColor) {
		[_fillColor setFill];
		[path fill];
	}

	if (shadow) {
		[NSGraphicsContext restoreGraphicsState];
	}
	
	if (_borderColor) {
		[_borderColor setStroke];
		[path setLineWidth:_borderWidth];
		[path stroke];
	}
	
	// Now make sure sub views can not draw beyond the view border.
	// Unfortunately there seems to be no official method to do this nor any documentation.
	// But the trick below works fine.
	// Note: but some Cocoa controls seem to draw outside of update, e.g. scroll bars. So this
	// is not 100% solid.

	// Restore the save state which is done by the AppKit around - [NSView drawRect:]
	[NSGraphicsContext restoreGraphicsState];
	// Establish clip for sub views.
	[path addClip];
	// Save the state again, now including the new clip. After returning from this function,
	// the AppKit will restore the state again, but the clip will survive.
	[NSGraphicsContext saveGraphicsState];
}

#pragma mark NSCoding Protocol --------------------------------------------------------------
//-------------------------------------------------------------------------------------------
- (id) initWithCoder:(NSCoder*)decoder {
	NSAssert ([decoder allowsKeyedCoding], @"oops, got a sequential archive");

	self = [super initWithCoder:decoder];
	if (self) {
		// Note: can decode keys in any order
		_cornerRadius = [decoder decodeFloatForKey:@"SP_cornerRadius"];

		_fillColor = [[decoder decodeObjectForKey:@"SP_fillColor"] retain];
		_gradient = [[decoder decodeObjectForKey:@"SP_gradient"] retain];
		
		_borderColor = [[decoder decodeObjectForKey:@"SP_borderColor"] retain];
		_borderWidth = [decoder decodeFloatForKey:@"SP_borderWidth"];
		
		_boxShadow = [[decoder decodeObjectForKey:@"SP_shadow"] retain];
	}
	
	return self;
}

//-------------------------------------------------------------------------------------------
- (void) encodeWithCoder:(NSCoder*)encoder {
	NSAssert ([encoder allowsKeyedCoding], @"oops, got a sequential archive");

   [super encodeWithCoder:encoder];
	
	[encoder encodeFloat:_cornerRadius forKey:@"SP_cornerRadius"];

	[encoder encodeObject:_fillColor forKey:@"SP_fillColor"];
	[encoder encodeObject:_gradient forKey:@"SP_gradient"];

	[encoder encodeObject:_borderColor forKey:@"SP_borderColor"];
	[encoder encodeFloat:_borderWidth forKey:@"SP_borderWidth"];

	[encoder encodeObject:_boxShadow forKey:@"SP_shadow"];
}


@end
